# 機能設計書 2-next build

## 概要

本ドキュメントは、Next.jsのプロダクション用ビルドコマンド `next build` の機能設計を記述する。アプリケーションを本番環境向けに最適化してビルドするCLIコマンドであり、静的最適化・コード分割・バンドル生成を行う。

### 本機能の処理概要

`next build` コマンドは、Next.jsアプリケーションのプロダクションビルドを実行し、最適化されたバンドルを生成するCLIコマンドである。

**業務上の目的・背景**：Webアプリケーションのデプロイには、コードの最適化・バンドル・静的ページの事前生成が必要である。`next build` はこれらすべてを自動的に行い、高速なページロードと効率的なリソース配信を実現する。開発コードからプロダクションに適したアーティファクトを生成する工程の中核を担う。

**機能の利用シーン**：アプリケーションのデプロイ前に実行する。CI/CDパイプラインで自動実行されるケースが多い。ローカルでのビルド検証にも使用される。

**主要な処理内容**：
1. プロジェクトディレクトリの検証とバンドラー選択（Turbopack/Webpack/Rspack）
2. 設定の読み込みとバリデーション
3. TypeScriptの型チェック
4. ページ・ルートの解析と静的/動的の判定
5. Webpack/Turbopack/Rspackによるバンドル生成
6. 静的ページのプリレンダリング（SSG/ISR）
7. マニフェストファイルの生成
8. ビルドトレースによる依存関係追跡
9. ビルド結果の出力とテレメトリの記録

**関連システム・外部連携**：Turbopack/Webpack/Rspackバンドラー、SWCコンパイラ、テレメトリサーバー

**権限による制御**：特段の権限制御はない。ファイルシステムへの書き込み権限が必要。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 7 | 500エラーページ (App Error) | 補助機能 | ビルド時に静的500.htmlとして生成される処理 |

## 機能種別

CLIコマンド / ビルド処理 / 最適化

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| directory | string | No | プロジェクトディレクトリ。未指定時はカレントディレクトリ | 存在するディレクトリであること |
| --experimental-analyze | boolean | No | バンドル分析出力（Turbopack専用） | Turbopack使用時のみ有効 |
| --debug, -d | boolean | No | 詳細なビルド出力 | なし |
| --debug-prerender | boolean | No | プリレンダリングのデバッグモード | なし |
| --no-mangling | boolean | No | マングリング無効化 | なし |
| --profile | boolean | No | React本番プロファイリング有効化 | なし |
| --turbo / --turbopack | boolean | No | Turbopackでビルド | --webpackと排他 |
| --webpack | boolean | No | Webpackでビルド | --turbo/--turbopackと排他 |
| --experimental-app-only | boolean | No | App Routerルートのみビルド | なし |
| --experimental-build-mode | string | No | ビルドモード（default/compile/generate/generate-env） | choices指定 |
| --experimental-debug-memory-usage | boolean | No | メモリ使用量のデバッグ | なし |
| --experimental-upload-trace | string | No | トレースアップロード先URL | 有効なURL |
| --debug-build-paths | string | No | 選択的ビルドのglobパターン | カンマ区切りのglobパターン |
| --experimental-cpu-prof | boolean | No | CPUプロファイリング有効化 | なし |

### 入力データソース

- CLI引数（Commander.jsで解析）
- next.config.js / next.config.ts（設定ファイル）
- 環境変数（NEXT_DEBUG_BUILD等）
- プロジェクトのソースコード（pages/, app/, public/等）

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| .next/ ディレクトリ | directory | ビルドアーティファクト一式 |
| ビルドマニフェスト | JSON | ページとアセットのマッピング情報 |
| ルートマニフェスト | JSON | ルーティング情報（リライト、リダイレクト等） |
| 静的HTML | HTML files | プリレンダリングされた静的ページ |
| バンドルファイル | JS/CSS | 最適化・分割されたバンドル |
| ビルド結果サマリー | text | 各ルートのサイズと種別（stdout表示） |

### 出力先

- .next/ ディレクトリ（ビルドアーティファクト）
- 標準出力（ビルド進捗・結果サマリー）
- テレメトリサーバー（匿名使用統計）
- トレースアップロード先（オプション）

## 処理フロー

### 処理シーケンス

```
1. シグナルハンドラの登録（SIGTERM/SIGINT）
   └─ CPUプロファイル保存とプロセス終了
2. オプション解析とバンドラー選択
   └─ parseBundlerArgsによるTurbopack/Webpack/Rspackの決定
3. プロジェクトディレクトリ検証
   └─ getProjectDirでディレクトリ解決、existsSyncで存在確認
4. 選択的ビルドパスの解決（オプション）
   └─ --debug-build-paths指定時、globパターンからapp/pagesパスを解決
5. build()関数の呼び出し
   └─ src/build/index.tsのメインビルド処理に全パラメータを渡す
6. エラーハンドリング
   └─ 既知のエラーコード（WEBPACK_ERRORS等）はメッセージのみ表示
   └─ 未知のエラーはスタックトレースを表示
7. メモリデバッグモードのクリーンアップ
   └─ experimentalDebugMemoryUsage有効時に無効化
```

### フローチャート

```mermaid
flowchart TD
    A[next build 実行] --> B[シグナルハンドラ登録]
    B --> C[オプション解析・バンドラー選択]
    C --> D{--experimental-analyze && !Turbopack?}
    D -->|Yes| E[エラー終了]
    D -->|No| F[プロジェクトディレクトリ検証]
    F --> G{ディレクトリ存在?}
    G -->|No| H[エラー終了]
    G -->|Yes| I{debug-build-paths?}
    I -->|Yes| J[ビルドパス解決]
    I -->|No| K[build関数呼び出し]
    J --> K
    K --> L{ビルド成功?}
    L -->|Yes| M[プロセス終了 code:0]
    L -->|No| N{既知エラー?}
    N -->|Yes| O[エラーメッセージ表示]
    N -->|No| P[スタックトレース表示]
    O --> Q[プロセス終了 code:1]
    P --> Q
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-01 | バンドル分析の制約 | --experimental-analyzeはTurbopackでのみ使用可能 | --experimental-analyze指定時 |
| BR-02 | マングリング無効化の警告 | --no-mangling指定時にパフォーマンス影響の警告を表示 | --no-mangling指定時 |
| BR-03 | プロファイリングの警告 | --profile指定時にパフォーマンス影響の警告を表示 | --profile指定時 |
| BR-04 | デバッグプリレンダリングの警告 | 本番環境では使用しない旨の警告表示 | --debug-prerender指定時 |
| BR-05 | 既知エラーコードの処理 | INVALID_RESOLVE_ALIAS, WEBPACK_ERRORS, BUILD_OPTIMIZATION_FAILED, NEXT_EXPORT_ERROR, NEXT_STATIC_GEN_BAILOUT, EDGE_RUNTIME_UNSUPPORTED_APIの場合はメッセージのみ表示 | ビルドエラー発生時 |
| BR-06 | ビルド完了後の強制終了 | ビルド成功時はprocess.exit(0)で強制終了（オープンハンドルによるハング防止） | ビルド成功時 |

### 計算ロジック

特段の計算ロジックはない。ビルド内部の最適化計算はsrc/build/index.tsに委譲される。

## データベース操作仕様

### 操作別データベース影響一覧

本機能はデータベースを直接操作しない。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| INVALID_RESOLVE_ALIAS | ビルドエラー | モジュール解決エイリアスが無効 | next.config.jsの設定を確認 |
| WEBPACK_ERRORS | ビルドエラー | Webpackコンパイルエラー | エラーメッセージに従いコードを修正 |
| BUILD_OPTIMIZATION_FAILED | ビルドエラー | ビルド最適化失敗 | エラーメッセージを確認 |
| NEXT_EXPORT_ERROR | エクスポートエラー | 静的エクスポート時のエラー | 動的機能の使用を見直す |
| NEXT_STATIC_GEN_BAILOUT | 静的生成エラー | 静的生成からのベイルアウト | 動的レンダリングの使用を検討 |
| EDGE_RUNTIME_UNSUPPORTED_API | ランタイムエラー | Edge Runtimeで未サポートAPIの使用 | Node.js APIへの置き換え |

### リトライ仕様

本機能にリトライ仕様はない。エラー時はプロセスを終了する。

## トランザクション仕様

本機能にはデータベーストランザクションはない。ファイルシステムへの書き込みは.next/ディレクトリに対して行われ、ビルド開始時に既存のキャッシュがクリーンアップされる。

## パフォーマンス要件

- ビルド時間はプロジェクトの規模とバンドラー選択に依存する
- Turbopackはインクリメンタルビルドにより高速化を実現
- --experimental-debug-memory-usageでメモリ使用量を監視可能

## セキュリティ考慮事項

- `--experimental-upload-trace` は機密データを含むため、信頼できるURLのみに使用する
- ビルド出力に環境変数の値が含まれる場合があるため、公開リポジトリへのコミットに注意

## 備考

- ビルド成功後は`process.exit(0)`で強制終了する（オープンハンドル/コネクションによるプロセスハングを防止）
- --experimental-build-modeの`compile`と`generate`はビルドフェーズを分離して実行可能
- メモリデバッグモードはfinally句で必ず無効化される

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | next-build.ts | `packages/next/src/cli/next-build.ts` | NextBuildOptions型定義（19-36行目）でビルドオプションの全体像を把握 |
| 1-2 | bundler.ts | `packages/next/src/lib/bundler.ts` | Bundler列挙型（2-6行目）でバンドラーの種類を理解 |

**読解のコツ**: NextBuildOptions型は多くのexperimental系オプションを含む。`experimentalBuildMode`は4つの文字列リテラル型で定義されている点に注意。

#### Step 2: エントリーポイントを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | next.ts | `packages/next/src/bin/next.ts` | buildコマンドの定義（142-220行目）、オプションの定義とアクション |
| 2-2 | next-build.ts | `packages/next/src/cli/next-build.ts` | nextBuild関数（38-164行目）がメインエントリーポイント |

**主要処理フロー**:
1. **39-46行目**: SIGTERM/SIGINTハンドラの登録（CPUプロファイル保存）
2. **66-72行目**: バンドラー選択とexperimental-analyzeの互換性チェック
3. **99-103行目**: プロジェクトディレクトリの検証
4. **108-124行目**: 選択的ビルドパスの解決（--debug-build-paths）
5. **126-138行目**: build()関数の呼び出し（10個の引数を渡す）
6. **139-163行目**: エラーハンドリングとメモリデバッグモードのクリーンアップ

#### Step 3: ビルドコア処理を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | index.ts | `packages/next/src/build/index.ts` | build()関数のメイン処理。設定読み込み、ページ解析、バンドル生成を統合 |
| 3-2 | resolve-build-paths.ts | `packages/next/src/lib/resolve-build-paths.ts` | 選択的ビルドパスの解決ロジック |

**主要処理フロー**:
- **build/index.ts**: 1000行以上の大規模関数。主要フェーズ：設定読み込み → TypeScriptセットアップ → ページ収集 → バンドル生成 → 静的生成 → マニフェスト出力

### プログラム呼び出し階層図

```
bin/next.ts (CLIエントリーポイント)
    │
    ├─ cli/next-build.ts::nextBuild()
    │      ├─ lib/bundler.ts::parseBundlerArgs()
    │      ├─ lib/get-project-dir.ts::getProjectDir()
    │      ├─ lib/resolve-build-paths.ts::resolveBuildPaths() [オプション]
    │      ├─ lib/memory/startup.ts::enableMemoryDebuggingMode() [オプション]
    │      ├─ build/index.ts::build()
    │      │      ├─ server/config.ts::loadConfig()
    │      │      ├─ lib/find-pages-dir.ts::findPagesDir()
    │      │      ├─ lib/verify-typescript-setup.ts
    │      │      ├─ build/webpack-config.ts [Webpack時]
    │      │      ├─ turbopack/ [Turbopack時]
    │      │      ├─ build/static-paths/ (SSG生成)
    │      │      └─ build/manifests/ (マニフェスト生成)
    │      └─ lib/memory/shutdown.ts::disableMemoryDebuggingMode() [オプション]
    │
    └─ server/lib/cpu-profile.ts::saveCpuProfile() [SIGTERM/SIGINT時]
```

### データフロー図

```
[入力]                    [処理]                         [出力]

CLI引数 ─────────────▶ next.ts (Commander解析) ──────▶ NextBuildOptions
                              │
NextBuildOptions ────▶ nextBuild() ──────────────────▶ build()
                              │                            │
next.config.js ──────▶ loadConfig() ────────────────▶ NextConfigComplete
                              │
ソースコード ────────▶ バンドラー処理 ─────────────────▶ .next/ ディレクトリ
  (pages/, app/)             │                            ├─ バンドルファイル
                              │                            ├─ 静的HTML
                              │                            └─ マニフェスト
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| next.ts | `packages/next/src/bin/next.ts` | ソース | CLIメインエントリーポイント |
| next-build.ts | `packages/next/src/cli/next-build.ts` | ソース | next buildコマンドのメイン処理 |
| index.ts | `packages/next/src/build/index.ts` | ソース | ビルドコア処理 |
| bundler.ts | `packages/next/src/lib/bundler.ts` | ソース | バンドラー選択ロジック |
| get-project-dir.ts | `packages/next/src/lib/get-project-dir.ts` | ソース | プロジェクトディレクトリ解決 |
| resolve-build-paths.ts | `packages/next/src/lib/resolve-build-paths.ts` | ソース | 選択的ビルドパスの解決 |
| cpu-profile.ts | `packages/next/src/server/lib/cpu-profile.ts` | ソース | CPUプロファイリング |
| startup.ts | `packages/next/src/lib/memory/startup.ts` | ソース | メモリデバッグモード開始 |
| shutdown.ts | `packages/next/src/lib/memory/shutdown.ts` | ソース | メモリデバッグモード終了 |
| utils.ts | `packages/next/src/server/lib/utils.ts` | ソース | ユーティリティ関数 |
| webpack-config.ts | `packages/next/src/build/webpack-config.ts` | ソース | Webpack設定生成 |
